jQuery(document).ready(function($) {
    
    // Button to generate content
    $('#aich-content-form').on('submit', function(e) {

        e.preventDefault();
        const provider = $('#ai-provider').val();
        const prompt = $('#content-prompt').val().trim();
        const type = $('#content-type').val();
        
        if (!prompt) {
            alert(ticlabsaich_data.strings.enter_prompt);
            return;
        }
        
        $('#aich-result').hide();
        $('#aich-error').hide();
        $('#aich-loading').show();
        $('#generate-content-btn').prop('disabled', true);
        
        $.ajax({
            url: ticlabsaich_data.ajax_url,
            type: 'POST',
            data: {
                action: 'ticlabsaich_get_text_api_config',
                nonce: ticlabsaich_data.nonce,
                provider: provider,
                prompt: prompt,
                type: type
            },
            success: function(response) {
                if (response.success) {

                    const url = response.data.url;
                    const apiKey = response.data.api_key;
                    const system_content = response.data.system_content;
                    const user_content = response.data.user_content;
                    const model = response.data.model;
                    const max_tokens = response.data.max_tokens || 15000;
                    const domain = response.data.domain;
                    const wait_time = response.data.waiting_time || "45-60";
                    
                    $('#wait-time').text(wait_time);

                    ticlabsaich_makeAIProviderRequest(provider, url, apiKey, model, system_content, user_content, max_tokens, domain);
                    
                } else {
                    ticlabsaich_showError('Error: ' + (response.data.message || 'Unknown error'));
                    $('#aich-loading').hide();
                    $('#generate-content-btn').prop('disabled', false);
                }
            },
            error: function() {
                $('#aich-loading').hide();
                $('#generate-content-btn').prop('disabled', false);
                ticlabsaich_showError(ticlabsaich_data.strings.request_failed);
            }
        });

    });
    
    // Copiar el contenido generado al portapapeles
    $('#copy-content-btn').on('click', function() {
        const content = $('#aich-result').data('content');
        
        navigator.clipboard.writeText(content).then(function() {
            alert(ticlabsaich_data.strings.copied_to_clipboard);
        }).catch(function() {
            alert(ticlabsaich_data.strings.copy_failed);
        });
    });
    
    // Button to generate image
    $('#aich-image-form').on('submit', function(e) {

        e.preventDefault();
        const provider = $('#ai-image-provider').val();
        const prompt = $('#image-prompt').val().trim();
        const size = $('#image-size').val();
        
        if (!prompt) {
            alert(ticlabsaich_data.strings.enter_description);
            return;
        }
        
        $('#aich-image-result').hide();
        $('#aich-image-error').hide();
        $('#aich-image-loading').show();
        $('#generate-image-btn').prop('disabled', true);
        
        $.ajax({
            url: ticlabsaich_data.ajax_url,
            type: 'POST',
            data: {
                action: 'ticlabsaich_get_image_api_config',
                nonce: ticlabsaich_data.nonce,
                provider: provider,                
            },
            success: function(response) {
                
                if (response.success) {

                    const url = response.data.url;
                    const get = response.data.get;
                    const apiKey = response.data.api_key;                    
                    const model = response.data.model;                    
                    const domain = response.data.domain;
                    const size_ = response.data.sizes[size] || null;
                    const wait_time = response.data.waiting_time || "45-60";

                    $('#wait-time').text(wait_time);

                    ticlabsaich_makeImageProviderRequest(provider, url, apiKey, model, prompt, size_, domain, get);

                } else {
                    ticlabsaich_showImageError('Error: ' + (response.data.message || 'Unknown error'));
                    $('#aich-image-loading').hide();
                    $('#generate-image-btn').prop('disabled', false);
                }
            },
            error: function() {
                $('#aich-image-loading').hide();
                $('#generate-image-btn').prop('disabled', false);
                ticlabsaich_showImageError(ticlabsaich_data.strings.request_failed);
            }
        });
    });
    
    // Click on tab button    
    $('div.aich-tab-button').on('click', function(e) {

        e.preventDefault();
        const target = e.currentTarget;
        
        const tab = $(target).data('tab');
        const tag = '#' + tab + '-tab-content';

        if($(tag).hasClass('hidden')) {
            $(tag).removeClass('hidden');
        } else {
            $(tag).addClass('hidden');
        }

    });
    
    function ticlabsaich_makeSyncImageRequest(provider, url, apiKey, model, prompt, size, domain) {
        
        const [contentType, data] = ticlabsaich_generate_image_data_body(provider, apiKey, model, prompt, size, domain);
        
        const headers = ticlabsaich_get_image_request_headers(provider, apiKey);

        $.ajax({
            url: url,
            type: 'POST',
            processData: (contentType === false)?false:true,
            contentType: contentType,
            headers: headers,
            data: data,
            timeout: 300000,
            success: function(response) {                
                ticlabsaich_save_image_response_to_media_library(provider, response, prompt);                                
            },
            error: function(xhr, status, error) {
                ticlabsaich_on_error_image_response();                
            }
        });
        
    }
    
    function ticlabsaich_save_image_to_media_library(url, prompt) {

        $.ajax({
            url: ticlabsaich_data.ajax_url,
            type: 'POST',
            data: {
                action: 'ticlabsaich_save_image_to_media_library',
                nonce: ticlabsaich_data.nonce,
                image_url: url,
                prompt: prompt
            },
            success: function(response) {                
                if (response.success) {                    
                    const media_url = response.data.media_url;
                    const attachment_id = response.data.attachment_id;
                    ticlabsaich_on_success_image_response(media_url, prompt);
                } else {
                    ticlabsaich_on_error_image_response(); 
                }
            },
            error: function() {
                ticlabsaich_on_error_image_response(); 
            }
        });
        
    }

    // Save API keys and model settings
$('#aich-settings-form').on('submit', function(e) {
    e.preventDefault();
    
    const openai_api_key = $('#openai-api-key').val().trim();
    const openai_api_model = $('#openai-api-model').val();

    let data = {
        action: 'ticlabsaich_save_api_key',
        nonce: ticlabsaich_data.nonce,
        "openai-api-key": openai_api_key,
        "openai-api-model": openai_api_model,
    };

    const providers = JSON.parse(ticlabsaich_data.providers);
    for (const [key, provider_data] of Object.entries(providers['text_generation'])) {
        if(provider_data['tier'] === 'pro') {
            const api_key = $('#' + key + '-api-key').val().trim();
            data[ key + '-api-key' ] = api_key;
        }
    }

    for (const [key, provider_data] of Object.entries(providers['image_generation'])) {
        if(provider_data['tier'] === 'pro') {
            const api_key = $('#' + key + '-api-key').val().trim();
            data[ key + '-api-key' ] = api_key;
        }
    }
            
    $('#save-settings-btn').prop('disabled', true);
    
    $.ajax({
        url: ticlabsaich_data.ajax_url,
        type: 'POST',
        data: data,
        success: function(response) {
            $('#save-settings-btn').prop('disabled', false);
            const $msg = $('#settings-message');
            
            if (response.success) {
                $msg.removeClass('notice-error').addClass('notice-success');
                $msg.html('<p>' + response.data.message + '</p>');
            } else {
                $msg.removeClass('notice-success').addClass('notice-error');
                $msg.html('<p>' + response.data.message + '</p>');
            }
            $msg.show();
            
            setTimeout(function() {
                $msg.fadeOut();
            }, 3000);
        },
        error: function() {
            $('#save-settings-btn').prop('disabled', false);
            alert(ticlabsaich_data.strings.request_failed);
        }
    });
});

// Save license key
$('#aich-license-form').on('submit', function(e) {
    e.preventDefault();

    const license_key = $('#aich_license').val().trim();
    
    if (!license_key) {
        alert(ticlabsaich_data.strings.enter_license_key);
        return;
    }
            
    $('#save-license-btn').prop('disabled', true);
    
    $.ajax({
        url: ticlabsaich_data.ajax_url,
        type: 'POST',
        data: {
            action: 'ticlabsaich_save_license_key',
            nonce: ticlabsaich_data.nonce,
            license_key: license_key,                
        },
        success: function(response) {
            $('#save-license-btn').prop('disabled', false);
            const $msg = $('#license-message');
            
            if (response.success) {
                $msg.removeClass('notice-error').addClass('notice-success');
                $msg.html('<p>' + response.data.message + '</p>');
            } else {
                $msg.removeClass('notice-success').addClass('notice-error');
                $msg.html('<p>' + response.data.message + '</p>');
            }
            $msg.show();
            
            setTimeout(function() {
                $msg.fadeOut();
                document.location.reload();
            }, 3000);
        },
        error: function() {
            $('#save-license-btn').prop('disabled', false);
            alert(ticlabsaich_data.strings.request_failed);
        }
    });
});

function ticlabsaich_makeAIProviderRequest(provider, url, apiKey, model, system_content, user_content, max_tokens, domain) {
        
    const openAIRequestType = [
        'openai', 'jamba', 'cohere', 'mistral', 'together', 'groq', 'perplexity', 'nebius', 'huggingface',
        'cerebras', 'upstage', 'baseten', 'sambanova', 'openrouter'
    ];
    const openAIStreamRequestType = ['fireworks'];
    const backendRequestType = ['proplus'];
    const anthropicRequestType = ['claude'];
    const geminiRequestType = ['gemini'];

    if (openAIRequestType.includes(provider)) {
        ticlabsaich_makeOpenAIRequest(provider, url, apiKey, model, system_content, user_content, max_tokens);
    } else if (backendRequestType.includes(provider)) {
        ticlabsaich_makeFreeProPlusAIRequest(url, apiKey, model, system_content, user_content, max_tokens, domain);
    } else if (anthropicRequestType.includes(provider)) {
        ticlabsaich_makeAnthropicAIRequest(provider, url, apiKey, model, system_content, user_content, max_tokens);
    } else if (geminiRequestType.includes(provider)) {
        ticlabsaich_makeGeminiAIRequest(provider, url, apiKey, model, system_content, user_content, max_tokens);
    } else if (openAIStreamRequestType.includes(provider)) {
        ticlabsaich_makeOpenAIStreamRequest(provider, url, apiKey, model, system_content, user_content, max_tokens);
    }
    
}

function ticlabsaich_makeImageProviderRequest(provider, url, apiKey, model, prompt, size, domain, get) {

    const syncRequestType = ['dalle3', 'stability', 'falai', 'img_huggingface', 'img_gemini', 'img_together', 'proplus'];
    const asyncRequestType = ['magichour', 'deapi', 'imaginepro'];

    if(syncRequestType.includes(provider)) {
        ticlabsaich_makeSyncImageRequest(provider, url, apiKey, model, prompt, size, domain);
    } else if(asyncRequestType.includes(provider)) {            
        ticlabsaich_makeAsyncImageRequest(provider, url, get, apiKey, model, prompt, size);
    }
    
}

function ticlabsaich_makeOpenAIRequest(provider, url, apiKey, model, system_content, user_content, max_tokens) {
    
    const maxTokensDataType = [
        'jamba', 'cohere', 'mistral', 'together', 'groq', 'perplexity', 'nebius', 'huggingface',
        'upstage', 'baseten', 'sambanova', 'openrouter'
    ];
    const maxCompletionTokensDataType = ['openai', 'cerebras'];

    let maxTokensData = '';
    if (maxTokensDataType.includes(provider)) {
        maxTokensData = 'max_tokens';
    } else if (maxCompletionTokensDataType.includes(provider)) {
        maxTokensData = 'max_completion_tokens';
    }

    let data = {
        model: model,
        messages: [
            { role: 'system', content: system_content },
            { role: 'user', content: user_content }
        ],            
    };
    data[maxTokensData] = max_tokens;

    $.ajax({
        url: url,
        type: 'POST',
        contentType: 'application/json',
        headers: {
            'Authorization': 'Bearer ' + apiKey,
            'Content-Type': 'application/json'
        },
        data: JSON.stringify(data),
        timeout: 300000,
        success: function(response) {

            const content = ticlabsaich_get_response_content(provider, response);
            ticlabsaich_on_success_ai_response(content);

        },
        error: function(xhr, status, error) {
            ticlabsaich_on_error_ai_response();                
        }
    });
}

function ticlabsaich_get_response_content(provider, response) {

    const openAIResponseType = [
        'openai', 'jamba', 'mistral', 'together', 'groq', 'perplexity', 'nebius', 'huggingface',
        'cerebras', 'fireworks', 'upstage', 'baseten', 'sambanova', 'openrouter'
    ];
    const anthropicResponseType = ['claude'];
    const geminiResponseType = ['gemini'];
    const cohereResponseType = ['cohere'];

    let content = '';

    if (openAIResponseType.includes(provider)) {
        content = response.choices[0].message.content;            
    } else if (anthropicResponseType.includes(provider)) {
        content = response.content[0].text;            
    } else if (geminiResponseType.includes(provider)) {
        content = response.candidates[0].content.parts[0].text;
    } else if (cohereResponseType.includes(provider)) {
        content = response.message.content[0].text;
    }

    content = content.replace('```html', '');
    content = content.replace('```', '');

    return content;

}

function ticlabsaich_makeAnthropicAIRequest(provider, url, apiKey, model, system_content, user_content, max_tokens) {
                
    $.ajax({
        url: url,
        type: 'POST',
        contentType: 'application/json',
        headers: {
            'x-api-key': apiKey,
            'anthropic-version': '2023-06-01',
            'anthropic-dangerous-direct-browser-access': 'true',
            'Content-Type': 'application/json'
        },
        data: JSON.stringify({
            model: model,
            system: system_content,
            messages: [                    
                { role: 'user', content: user_content }
            ],
            max_tokens: max_tokens,
        }),
        timeout: 300000,
        success: function(response) {
            
            const content = ticlabsaich_get_response_content(provider, response);
            ticlabsaich_on_success_ai_response(content);

        },
        error: function(xhr, status, error) {
            console.log(xhr, status, error);
            ticlabsaich_on_error_ai_response();                
        }
    });
}

function ticlabsaich_makeGeminiAIRequest(provider, url, apiKey, model, system_content, user_content, max_tokens) {
            
    $.ajax({
        url: `${url}${model}:generateContent?key=${apiKey}`,
        type: 'POST',
        contentType: 'application/json',
        headers: {                
            'Content-Type': 'application/json'
        },
        data: JSON.stringify({
            systemInstruction: {
                parts: [
                    {
                        text: system_content
                    }
                ]
            },
            contents: [
                {
                    role: 'user',
                    parts: [
                        {
                            text: user_content
                        }
                    ]
                }
            ],
            generationConfig: {
                temperature: 0.7,
                maxOutputTokens: max_tokens
            }
        }),
        timeout: 300000,
        success: function(response) {
            
            const content = ticlabsaich_get_response_content(provider, response);
            ticlabsaich_on_success_ai_response(content);

        },
        error: function(xhr, status, error) {
            console.log(xhr, status, error);
            ticlabsaich_on_error_ai_response();                
        }
    });
}
        
function ticlabsaich_makeOpenAIStreamRequest(provider, url, apiKey, model, system_content, user_content, max_tokens) {

    let fullContent = '';
    let lastIndex = 0;
    
    const maxTokensDataType = ['fireworks'];

    let maxTokensData = '';
    if (maxTokensDataType.includes(provider)) {
        maxTokensData = 'max_tokens';
    }

    let data = {
        stream: true,
        model: model,
        messages: [
            { role: 'system', content: system_content },
            { role: 'user', content: user_content }
        ],            
    };
    data[maxTokensData] = max_tokens;

    $.ajax({
        url: url,
        type: 'POST',
        contentType: 'application/json',
        headers: {
            'Authorization': `Bearer ${apiKey}`,
            'Content-Type': 'application/json'
        },
        data: JSON.stringify(data),
        timeout: 300000,
        xhr: function() {  

            const xhr = new window.XMLHttpRequest();
            
            xhr.addEventListener('progress', function(e) {

                if (xhr.status === 200 && xhr.responseText) {

                    const response = xhr.responseText;
                    const lines = response.split('\n');
                    
                    lines.forEach((line, index) => {

                        if (index < lastIndex) return; 
                        
                        if (line.startsWith('data: ')) {

                            const data = line.substring(6);
                            
                            if (data === '[DONE]') {                                    
                                return;
                            }
                            
                            try {

                                const json = JSON.parse(data);
                                
                                if (json.choices && json.choices[0] && json.choices[0].delta && json.choices[0].delta.content) {
                                    
                                    const chunk = json.choices[0].delta.content;
                                    fullContent += chunk;
                                    
                                }
                                lastIndex = index + 1;
                            } catch (e) {
                                console.error('Error parsing JSON:', e);
                            }
                        }
                    });
                }
            });
            
            return xhr;

        },
        success: function(response) {
            let content = fullContent.replace('```html', '');
            content = content.replace('```', '');           
            ticlabsaich_on_success_ai_response(content);
        },
        error: function(xhr, status, error) {
            console.log(xhr, status, error);
            ticlabsaich_on_error_ai_response();                
        }
    });
}

function ticlabsaich_makeFreeProPlusAIRequest(url, apiKey, model, system_content, user_content, max_tokens, domain) {
            
    $.ajax({
        url: url,
        type: 'POST',
        contentType: 'application/json',
        headers: {
            'Content-Type': 'application/json'
        },
        data: JSON.stringify({
            action: model,
            license: apiKey,
            domain: domain,
            system_content: system_content,
            user_content: user_content
        }),
        timeout: 300000,
        success: function(response) {

            let content = JSON.parse(response).content;

            content = content.replace('```html', '');
            content = content.replace('```', ''); 

            ticlabsaich_on_success_ai_response(content);

        },
        error: function(xhr, status, error) {
            console.log(xhr, status, error);
            ticlabsaich_on_error_ai_response();                
        }
    });
}

async function ticlabsaich_makeAsyncImageRequest(provider, url, get, apiKey, model, prompt, size) {
        
    const startResponse = await $.ajax({
        url: ticlabsaich_data.ajax_url,
        type: 'POST',
        data: {
            action: 'ticlabsaich_async_generate_image',
            nonce: ticlabsaich_data.nonce,
            provider: provider,
            url: url,
            get: get,
            api_key: apiKey,
            model: model,
            prompt: prompt,
            size: size
        },            
    });

    if(!startResponse.success) {
        ticlabsaich_on_error_image_response();  
        return;
    }

    const response_data = startResponse.data;
    
    return await ticlabsaich_pollStatus(provider, get, apiKey, prompt, response_data);
    
}

function ticlabsaich_get_image_request_headers(provider, apiKey) {

    const bearerType = ['dalle3', 'stability', 'img_huggingface', 'img_together'];
    const keyType = ['falai'];
    const geminiType = ['img_gemini'];
    const backendType = ['proplus'];

    if(bearerType.includes(provider)) {
        return {
            'Authorization': 'Bearer ' + apiKey,                
            'Accept': 'application/json'
        };
    } else if(keyType.includes(provider)) {
        return {
            'Authorization': 'Key ' + apiKey,                
            'Accept': 'application/json'
        }
    } else if(geminiType.includes(provider)) {
        return {
            'x-goog-api-key': apiKey,                
            'Accept': 'application/json'
        }
    } else if(backendType.includes(provider)) {
        return {                
            'Accept': 'application/json'
        }
    }

}

function ticlabsaich_generate_image_data_body(provider, apiKey, model, prompt, size, domain='') {

    const openAIDataType = ['dalle3', 'falai'];
    const stabilityFormDataType = ['stability'];
    const huggingfaceDataType = ['img_huggingface'];
    const geminiDataType = ['img_gemini'];
    const togetherDataType = ['img_together'];
    const backendDataType = ['proplus'];

    if(openAIDataType.includes(provider)) {
        return [
            'application/json', 
            JSON.stringify({
                model: model,
                prompt: prompt,
                size: size,
                n: 1,
                quality: 'standard'
            })
        ];
    } else if(stabilityFormDataType.includes(provider)) {
        const formData = new FormData();
        formData.append('model', model);
        formData.append('mode', 'text-to-image');
        formData.append('prompt', prompt);
        formData.append('aspect_ratio', size);
        formData.append('output_format', 'png');
        formData.append('style_preset', 'photographic');
        return [
            false,
            formData
        ];
    } else if(huggingfaceDataType.includes(provider)) {
        return [
            'application/json', 
            JSON.stringify({
                "inputs": prompt,
                "parameters": {
                    "width": size[0],
                    "height": size[1]
                }
            })
        ];
    } else if(geminiDataType.includes(provider)) {
        return [
            'application/json', 
            JSON.stringify({
                "contents": [{
                    "parts": [{
                        "text": prompt
                    }]
                }],
                "generationConfig": {
                    "temperature": 1.5,
                    "candidateCount": 1,
                    "imageConfig": {
                        "aspectRatio": size,
                        "imageSize": "2k"
                    }
                }
            })
        ];
    } else if(togetherDataType.includes(provider)) {
        return [
            'application/json', 
            JSON.stringify({
                "model": model,
                "prompt" : prompt,
                "width": size[0],
                "height": size[1],
                "steps": 28
            })
        ];
    } else if(backendDataType.includes(provider)) {
        return [
            'application/json', 
            JSON.stringify({
                "action": model,
                "license": apiKey,
                "domain": domain,
                "prompt": prompt,
                "size": size
            })
        ];
    }

}

function ticlabsaich_save_image_response_to_media_library(provider, response, prompt) {

    const urlResponseType = ['dalle3', 'falai', 'img_together', 'proplus'];
    const base64ResponseType = ['stability', 'img_huggingface', 'img_gemini'];

    if(urlResponseType.includes(provider)) {
        [url, revised_prompt] = ticlabsaich_get_url_and_revised_prompt(provider, response, prompt);
        ticlabsaich_save_image_to_media_library(url, revised_prompt);
    } else if(base64ResponseType.includes(provider)) {
        [b64ImageData, revised_prompt] = ticlabsaich_get_image_data_and_revised_prompt(provider, response, prompt);
        ticlabsaich_save_image_data_to_media_library(b64ImageData, revised_prompt);
    }

}

function ticlabsaich_get_url_and_revised_prompt(provider, response, prompt) {

    const openAIResponseType = ['dalle3', 'img_together'];
    const falaiResponseType = ['falai'];
    const backendResponseType = ['proplus'];
    

    if(openAIResponseType.includes(provider)) {
        return [
            response.data[0].url,
            response.data[0].revised_prompt || prompt
        ];
    } else if(falaiResponseType.includes(provider)) {
        return [
            response.images[0].url,
            prompt
        ];
    } else if(backendResponseType.includes(provider)) {
        const response_data = JSON.parse(response);
        return [
            response_data.url,
            response_data.revised_prompt || prompt
        ];
    } 

}

function ticlabsaich_get_image_data_and_revised_prompt(provider, response, prompt) {

    const stabilityResponseType = ['stability'];
    const huggingfaceResponseType = ['img_huggingface'];
    const geminiResponseType = ['img_gemini'];

    if(stabilityResponseType.includes(provider)) {
        return [
            response.image,
            prompt
        ]
    } else if(huggingfaceResponseType.includes(provider)) {
        return [
            response,
            prompt
        ];
    } else if(geminiResponseType.includes(provider)) {
        return [
            response.candidates[0].content.parts[0].inlineData.data,
            prompt
        ];
    }

}

async function ticlabsaich_pollStatus(provider, get, apiKey, prompt, response_data) {

    const maxAttempts = 60; // 2 minutos máximo
    let attempts = 0;
    
    while (attempts < maxAttempts) {

        await new Promise(resolve => setTimeout(resolve, 2000));
        
        const statusResponse = await $.ajax({
            url: ticlabsaich_data.ajax_url,
            type: 'POST',
            data: {
                action: 'ticlabsaich_async_check_status',
                nonce: ticlabsaich_data.nonce,                    
                provider: provider,
                get: get,
                api_key: apiKey,
                response_data: response_data
            }
        });
        
        if (!statusResponse.success) {
            ticlabsaich_on_error_image_response();
            return;
        }
        
        const statusData = statusResponse.data;
                    
        if (statusData.status === 'complete') {
                            
            const url_image = statusData.url;
            const revised_prompt = prompt;

            ticlabsaich_save_image_to_media_library(url_image, revised_prompt);                
            return;

        }
        
        if (statusData.status === 'failed') {
            ticlabsaich_on_error_image_response();
        }
        
        attempts++;

    }
    
    ticlabsaich_on_error_image_response();

}

function ticlabsaich_save_image_data_to_media_library(data, prompt) {

    $.ajax({
        url: ticlabsaich_data.ajax_url,
        type: 'POST',
        data: {
            action: 'ticlabsaich_save_image_data_to_media_library',
            nonce: ticlabsaich_data.nonce,
            image_data: data,
            prompt: prompt
        },
        success: function(response) {                
            if (response.success) {                    
                const media_url = response.data.media_url;
                const attachment_id = response.data.attachment_id;
                ticlabsaich_on_success_image_response(media_url, prompt);
            } else {
                    ticlabsaich_on_error_image_response(); 
            }
        },
        error: function() {
            ticlabsaich_on_error_image_response(); 
        }
    });
    
}

    function ticlabsaich_on_success_ai_response(content) {
        
        $('#aich-loading').hide();
        $('#generate-content-btn').prop('disabled', false);

        $('#aich-result-content').html('<pre>' + content + '</pre>');
        $('#aich-result').show();
        $('#aich-result').data('content', content);

    }

    function ticlabsaich_on_success_image_response(url, revised_prompt) {
        
        $('#aich-image-loading').hide();
        $('#generate-image-btn').prop('disabled', false);

        $('#aich-image-preview').html('<img src="' + url + '" style="max-width: 100%; height: auto; border: 1px solid #ddd; border-radius: 4px;" />');
        $('#aich-revised-prompt').html('<strong>Revised prompt:</strong> ' + revised_prompt);
        $('#aich-image-result').show();

    }

    function ticlabsaich_on_error_ai_response() {

        $('#aich-loading').hide();
        $('#generate-content-btn').prop('disabled', false);
        ticlabsaich_showError(ticlabsaich_data.strings.request_failed);

    }

    function ticlabsaich_on_error_image_response() {

        $('#aich-image-loading').hide();
        $('#generate-image-btn').prop('disabled', false);
        ticlabsaich_showImageError(ticlabsaich_data.strings.request_failed);

    }

    function ticlabsaich_showError(message) {
        $('#aich-error').html('<p>' + message + '</p>').show();
    }
    
    function ticlabsaich_showImageError(message) {
        $('#aich-image-error').html('<p>' + message + '</p>').show();
    }
});
